--
-- BuyableFloatationWheels
-- Specialization for Buyable Floatation Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
--
-- @edit		JoXXer
-- @date		06/11/11
--
-- @edit                Wolverine for use with vehicle without animatedWheel
-- @date                20/02/2012

BuyableFloatationWheels = {};

function BuyableFloatationWheels.prerequisitesPresent(specializations)
    return true;
end;

function BuyableFloatationWheels:load(xmlFile)

	self.floatationWheelTriggerCallback = BuyableFloatationWheels.floatationWheelTriggerCallback;
	self.wheelDelete = BuyableFloatationWheels.wheelDelete;
	self.assembleWheels = SpecializationUtil.callSpecializationsFunction("assembleWheels");
	self.disassembleWheels = SpecializationUtil.callSpecializationsFunction("disassembleWheels");

	self.checkString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.floatationWheels#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.floatationWheels#activationTrigger"));


	addTrigger(self.usingTrigger, "floatationWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);

	self.belts = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.floatationWheels#belts"));

    self.buyableFloatationWheels = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.floatationWheels.wheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;

		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.transFix = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#transFix"));
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.buyableFloatationWheels, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.floatationWheelsUser = nil;

	self.savedVehicleID = -1;
	self.shouldUpdateLater = false;
end;

function BuyableFloatationWheels:delete()
	if self.floatationWheelsUser ~= nil then
		self.floatationWheelsUser:onDisassembling(self);
		for _,floatationWheel in pairs(self.buyableFloatationWheels) do
			delete(floatationWheel.node);
			delete(floatationWheel.transFix);
		end;
		self.buyableFloatationWheels = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableFloatationWheels:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableFloatationWheels:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local floatationWheelsUser = networkGetObject(id);
		if floatationWheelsUser ~= nil then
			self:assembleWheels(floatationWheelsUser, true);
		else
			self.savedVehicleID = id;
			self.shouldUpdateLater = true;
		end;
	end;
end;

function BuyableFloatationWheels:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.floatationWheelsUser ~= nil then
		idToWrite = networkGetObjectId(self.floatationWheelsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableFloatationWheels:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableFloatationWheels:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableFloatationWheels:update(dt)
end;

function BuyableFloatationWheels:updateTick(dt)

	-- try to load saved floatationWheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then
				self:assembleWheels(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;

	-- Attach wheels on client if not attached on load
	if self.shouldUpdateLater then
		local floatationWheelsUser = networkGetObject(self.savedVehicleID);
		if floatationWheelsUser ~= nil then
			self:assembleWheels(floatationWheelsUser, true);
			self.shouldUpdateLater = false;
			self.savedVehicleID = -1;
		end;
	end;
end;

function BuyableFloatationWheels:draw()
end;

function BuyableFloatationWheels:floatationWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter or onStay then
			if vehicle.rootNode == otherId then
				if vehicle.buyableWheels ~= nil then
					if vehicle.buyableWheels.checkString == self.checkString then
						if self.floatationWheelsUser ~= nil then
							if self.floatationWheelsUser == vehicle then
								vehicle.buyableWheels.floatationwheelsInRange = self;
							end;
						else
							vehicle.buyableWheels.floatationwheelsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then
			if otherId == vehicle.rootNode then
				if vehicle.buyableWheels ~= nil then
					vehicle.buyableWheels.floatationwheelsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableFloatationWheels:assembleWheels(vehicle, noEventSend)
	BuyableWheelsAttachEvent.sendEvent(self, vehicle, noEventSend);

	if self.belts ~= nil then
		setVisibility(self.belts, false);
	end;
	if self.floatationWheelsUser == nil then
		if vehicle.buyableWheels ~= nil then
			self.floatationWheelsUser = vehicle;
			for l,floatationWheel in pairs(self.buyableFloatationWheels) do
				if l == 1 then
					unlink(floatationWheel.node);
					unlink(floatationWheel.transFix);
					link(vehicle.wheels[1].driveNode, floatationWheel.node);
				elseif l == 2 then
					unlink(floatationWheel.node);
					unlink(floatationWheel.transFix);
					link(vehicle.wheels[2].driveNode, floatationWheel.node);
				elseif l == 3 then
					unlink(floatationWheel.node);
					unlink(floatationWheel.transFix);
					link(vehicle.wheels[3].driveNode, floatationWheel.node);
				elseif l == 4 then
					unlink(floatationWheel.node);
					unlink(floatationWheel.transFix);
					link(vehicle.wheels[4].driveNode, floatationWheel.node);
				end;
			end;
			self.floatationWheelsUser:onAssembling(self);
		end;
	end;
end;

function BuyableFloatationWheels:disassembleWheels(vehicle, noEventSend)
	BuyableWheelsDetachEvent.sendEvent(self, vehicle, noEventSend);

	self.floatationWheelsUser:onDisassembling(self);

	if self.belts ~= nil then
		setVisibility(self.belts, true);
	end;

	for k,wheel in pairs(self.buyableFloatationWheels) do
		unlink(wheel.transFix);
		unlink(wheel.node);
		setRotation(wheel.transFix, 0,0,0);
		setTranslation(wheel.transFix, 0,0,0);
		link(wheel.savePosition, wheel.transFix);
		link(wheel.transFix, wheel.node);
	end;
	self.floatationWheelsUser = nil;
end;


function BuyableFloatationWheels:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoordsFloatations");
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};
		end;
	end;

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableFloatationWheels:getSaveAttributesAndNodes(nodeIdent)
    local attributes = nil;

	if self.floatationWheelsUser ~= nil then
		local x,y,z = getWorldTranslation(self.floatationWheelsUser.rootNode);
		attributes = 'attacherCoordsFloatations="'.. x .. " " .. y .. " " .. z ..'"';
	end;

    return attributes, nil;
end;
