--
-- BuyableFrontDualWheels
-- Specialization for Buyable Twin Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
--
-- @edit		JoXXer
-- @date		08/11/11
--
-- @edit                Wolverine for use with vehicle without animatedWheel
-- @date                20/02/2012


BuyableFrontDualWheels = {};

function BuyableFrontDualWheels.prerequisitesPresent(specializations)
    return true;
end;

function BuyableFrontDualWheels:load(xmlFile)

	self.frontDualsWheelTriggerCallback = BuyableFrontDualWheels.frontDualsWheelTriggerCallback;
	self.wheelDelete = BuyableFrontDualWheels.wheelDelete;
	self.assembleWheels = SpecializationUtil.callSpecializationsFunction("assembleWheels");
	self.disassembleWheels = SpecializationUtil.callSpecializationsFunction("disassembleWheels");

	self.checkString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.frontDualWheels#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.frontDualWheels#activationTrigger"));


	addTrigger(self.usingTrigger, "frontDualsWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);

	self.connectors1 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.frontDualWheels#connectors1"));
	self.connectors2 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.frontDualWheels#connectors2"));

    self.buyableFrontDualWheels = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.frontDualWheels.wheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;

		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.collision = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#collision"));
		-- collision bug fix: on initial wheel assembling no collision is set
		setRigidBodyType(wheel.collision, "Kinematic");
		setRigidBodyType(wheel.collision, "NoRigidBody");
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.buyableFrontDualWheels, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.frontDualWheelsUser = nil;

	self.savedVehicleID = -1;
	self.shouldUpdateLater = false;
end;

function BuyableFrontDualWheels:delete()
	if self.frontDualWheelsUser ~= nil then
		self.frontDualWheelsUser:onDisassembling(self);
		for _,frontDualWheel in pairs(self.buyableFrontDualWheels) do
			delete(frontDualWheel.node);
			delete(frontDualWheel.collision);
		end;
		self.buyableFrontDualWheels = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableFrontDualWheels:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableFrontDualWheels:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local frontDualWheelsUser = networkGetObject(id);
		if frontDualWheelsUser ~= nil then
			self:assembleWheels(frontDualWheelsUser, true);
		else
			self.savedVehicleID = id;
			self.shouldUpdateLater = true;
		end;
	end;
end;

function BuyableFrontDualWheels:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.frontDualWheelsUser ~= nil then
		idToWrite = networkGetObjectId(self.frontDualWheelsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableFrontDualWheels:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableFrontDualWheels:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableFrontDualWheels:update(dt)
end;

function BuyableFrontDualWheels:updateTick(dt)

	-- try to load saved frontDualWheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then
				self:assembleWheels(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;

	-- Attach wheels on client if not attached on load
	if self.shouldUpdateLater then
		local frontDualWheelsUser = networkGetObject(self.savedVehicleID);
		if frontDualWheelsUser ~= nil then
			self:assembleWheels(frontDualWheelsUser, true);
			self.shouldUpdateLater = false;
			self.savedVehicleID = -1;
		end;
	end;
end;

function BuyableFrontDualWheels:draw()
end;

function BuyableFrontDualWheels:frontDualsWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter or onStay then
			if vehicle.rootNode == otherId then
				if vehicle.buyableWheels ~= nil then
					if vehicle.buyableWheels.checkString == self.checkString then
						if self.frontDualWheelsUser ~= nil then
							if self.frontDualWheelsUser == vehicle then
								vehicle.buyableWheels.frontDualwheelsInRange = self;
							end;
						else
							vehicle.buyableWheels.frontDualwheelsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then
			if otherId == vehicle.rootNode then
				if vehicle.buyableWheels ~= nil then
					vehicle.buyableWheels.frontDualwheelsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableFrontDualWheels:assembleWheels(vehicle, noEventSend)
	BuyableWheelsAttachEvent.sendEvent(self, vehicle, noEventSend);

	if self.connectors1 ~= nil then
		setVisibility(self.connectors1, true);
	end;
	if self.connectors2 ~= nil then
		setVisibility(self.connectors2, true);
	end;
	if self.frontDualWheelsUser == nil then
		if vehicle.buyableWheels ~= nil then
			self.frontDualWheelsUser = vehicle;
			--for k,wheel in pairs(vehicle.wheels) do
				for l,frontDualWheel in pairs(self.buyableFrontDualWheels) do
					if l == 1 then
						unlink(frontDualWheel.node);
						unlink(frontDualWheel.collision);
						link(vehicle.wheels[1].driveNode, frontDualWheel.node);
						--link(getParent(wheel.repr), frontDualWheel.collision);
						local x,y,z = getWorldTranslation(vehicle.wheels[1].driveNode);
						x,y,z = worldToLocal(getParent(frontDualWheel.collision), x,y,z);
						--setTranslation(frontDualWheel.collision,x,y,z);
						--setRigidBodyType(frontDualWheel.collision, "Kinematic");
					elseif l == 2 then
						unlink(frontDualWheel.node);
						unlink(frontDualWheel.collision);
						link(vehicle.wheels[2].driveNode, frontDualWheel.node);
						--link(getParent(wheel.repr), frontDualWheel.collision);
						local x,y,z = getWorldTranslation(vehicle.wheels[2].driveNode);
						x,y,z = worldToLocal(getParent(frontDualWheel.collision), x,y,z);
						--setTranslation(frontDualWheel.collision,x,y,z);
						--setRigidBodyType(frontDualWheel.collision, "Kinematic");
					end;
				end;
			--end;
			self.frontDualWheelsUser:onAssembling(self);
		end;
	end;
end;

function BuyableFrontDualWheels:disassembleWheels(vehicle, noEventSend)
	BuyableWheelsDetachEvent.sendEvent(self, vehicle, noEventSend);

	self.frontDualWheelsUser:onDisassembling(self);

	if self.connectors1 ~= nil then
		setVisibility(self.connectors1, false);
	end;
	if self.connectors2 ~= nil then
		setVisibility(self.connectors2, false);
	end;

	for k,wheel in pairs(self.buyableFrontDualWheels) do
		setRigidBodyType(wheel.collision, "NoRigidBody");
		unlink(wheel.collision);
		unlink(wheel.node);
		setRotation(wheel.collision, 0,0,0);
		setTranslation(wheel.collision, 0,0,0);
		link(wheel.savePosition, wheel.collision);
		link(wheel.collision, wheel.node);
	end;
	self.frontDualWheelsUser = nil;
end;


function BuyableFrontDualWheels:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoordsFrontDuals");
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};
		end;
	end;

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableFrontDualWheels:getSaveAttributesAndNodes(nodeIdent)
    local attributes = nil;

	if self.frontDualWheelsUser ~= nil then
		local x,y,z = getWorldTranslation(self.frontDualWheelsUser.rootNode);
		attributes = 'attacherCoordsFrontDuals="'.. x .. " " .. y .. " " .. z ..'"';
	end;

    return attributes, nil;
end;




