--
-- BuyableRearDualWheels
-- Specialization for Buyable Twin Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
--
-- @edit		JoXXer
-- @date		08/11/11
--
-- @edit                Wolverine for use with vehicle without animatedWheel
-- @date                20/02/2012


BuyableRearDualWheels = {};

function BuyableRearDualWheels.prerequisitesPresent(specializations)
    return true;
end;

function BuyableRearDualWheels:load(xmlFile)

	self.rearDualsWheelTriggerCallback = BuyableRearDualWheels.rearDualsWheelTriggerCallback;
	self.wheelDelete = BuyableRearDualWheels.wheelDelete;
	self.assembleWheels = SpecializationUtil.callSpecializationsFunction("assembleWheels");
	self.disassembleWheels = SpecializationUtil.callSpecializationsFunction("disassembleWheels");

	self.checkString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.rearDualWheels#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.rearDualWheels#activationTrigger"));


	addTrigger(self.usingTrigger, "rearDualsWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);

	self.connectors1 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.rearDualWheels#connectors1"));
	self.connectors2 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.rearDualWheels#connectors2"));

    self.buyableRearDualWheels = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.rearDualWheels.wheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;

		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.collision = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#collision"));
		-- collision bug fix: on initial wheel assembling no collision is set
		setRigidBodyType(wheel.collision, "Kinematic");
		setRigidBodyType(wheel.collision, "NoRigidBody");
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.buyableRearDualWheels, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.rearDualWheelsUser = nil;

	self.savedVehicleID = -1;
	self.shouldUpdateLater = false;
end;

function BuyableRearDualWheels:delete()
	if self.rearDualWheelsUser ~= nil then
		self.rearDualWheelsUser:onDisassembling(self);
		for _,rearDualWheel in pairs(self.buyableRearDualWheels) do
			delete(rearDualWheel.node);
			delete(rearDualWheel.collision);
		end;
		self.buyableRearDualWheels = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableRearDualWheels:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableRearDualWheels:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local rearDualWheelsUser = networkGetObject(id);
		if rearDualWheelsUser ~= nil then
			self:assembleWheels(rearDualWheelsUser, true);
		else
			self.savedVehicleID = id;
			self.shouldUpdateLater = true;
		end;
	end;
end;

function BuyableRearDualWheels:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.rearDualWheelsUser ~= nil then
		idToWrite = networkGetObjectId(self.rearDualWheelsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableRearDualWheels:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableRearDualWheels:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableRearDualWheels:update(dt)
end;

function BuyableRearDualWheels:updateTick(dt)

	-- try to load saved rearDualWheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then
				self:assembleWheels(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;

	-- Attach wheels on client if not attached on load
	if self.shouldUpdateLater then
		local rearDualWheelsUser = networkGetObject(self.savedVehicleID);
		if rearDualWheelsUser ~= nil then
			self:assembleWheels(rearDualWheelsUser, true);
			self.shouldUpdateLater = false;
			self.savedVehicleID = -1;
		end;
	end;
end;

function BuyableRearDualWheels:draw()
end;

function BuyableRearDualWheels:rearDualsWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter or onStay then
			if vehicle.rootNode == otherId then
				if vehicle.buyableWheels ~= nil then
					if vehicle.buyableWheels.checkString == self.checkString then
						if self.rearDualWheelsUser ~= nil then
							if self.rearDualWheelsUser == vehicle then
								vehicle.buyableWheels.rearDualwheelsInRange = self;
							end;
						else
							vehicle.buyableWheels.rearDualwheelsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then
			if otherId == vehicle.rootNode then
				if vehicle.buyableWheels ~= nil then
					vehicle.buyableWheels.rearDualwheelsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableRearDualWheels:assembleWheels(vehicle, noEventSend)
	BuyableWheelsAttachEvent.sendEvent(self, vehicle, noEventSend);

	if self.connectors1 ~= nil then
		setVisibility(self.connectors1, true);
	end;
	if self.connectors2 ~= nil then
		setVisibility(self.connectors2, true);
	end;
	if self.rearDualWheelsUser == nil then
		if vehicle.buyableWheels ~= nil then
			self.rearDualWheelsUser = vehicle;
			--for k,wheel in pairs(vehicle.wheels) do
				for l,rearDualWheel in pairs(self.buyableRearDualWheels) do
					if l == 1 then
						unlink(rearDualWheel.node);
						unlink(rearDualWheel.collision);
						link(vehicle.wheels[3].driveNode, rearDualWheel.node);
						link(getParent(vehicle.wheels[3].repr), rearDualWheel.collision);
						local x,y,z = getWorldTranslation(vehicle.wheels[3].driveNode);
						x,y,z = worldToLocal(getParent(rearDualWheel.collision), x,y,z);
						setTranslation(rearDualWheel.collision,x,y,z);
						--setRigidBodyType(rearDualWheel.collision, "Kinematic");
					elseif l == 2 then
						unlink(rearDualWheel.node);
						unlink(rearDualWheel.collision);
						link(vehicle.wheels[4].driveNode, rearDualWheel.node);
						link(getParent(vehicle.wheels[4].repr), rearDualWheel.collision);
						local x,y,z = getWorldTranslation(vehicle.wheels[4].driveNode);
						x,y,z = worldToLocal(getParent(rearDualWheel.collision), x,y,z);
						setTranslation(rearDualWheel.collision,x,y,z);
						--setRigidBodyType(rearDualWheel.collision, "Kinematic");
					end;
				end;
			--end;
			self.rearDualWheelsUser:onAssembling(self);
		end;
	end;
end;

function BuyableRearDualWheels:disassembleWheels(vehicle, noEventSend)
	BuyableWheelsDetachEvent.sendEvent(self, vehicle, noEventSend);

	self.rearDualWheelsUser:onDisassembling(self);

	if self.connectors1 ~= nil then
		setVisibility(self.connectors1, false);
	end;
	if self.connectors2 ~= nil then
		setVisibility(self.connectors2, false);
	end;

	for k,wheel in pairs(self.buyableRearDualWheels) do
		setRigidBodyType(wheel.collision, "NoRigidBody");
		unlink(wheel.collision);
		unlink(wheel.node);
		setRotation(wheel.collision, 0,0,0);
		setTranslation(wheel.collision, 0,0,0);
		link(wheel.savePosition, wheel.collision);
		link(wheel.collision, wheel.node);
	end;
	self.rearDualWheelsUser = nil;
end;


function BuyableRearDualWheels:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoordsRearDuals");
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};
		end;
	end;

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableRearDualWheels:getSaveAttributesAndNodes(nodeIdent)
    local attributes = nil;

	if self.rearDualWheelsUser ~= nil then
		local x,y,z = getWorldTranslation(self.rearDualWheelsUser.rootNode);
		attributes = 'attacherCoordsRearDuals="'.. x .. " " .. y .. " " .. z ..'"';
	end;

    return attributes, nil;
end;




