--
-- BuyableStandardWheels
-- Specialization for Buyable Standard Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
--
-- @edit		JoXXer
-- @date		06/11/11
--
-- @edit                Wolverine for use with vehicle without animatedWheel
-- @date                20/02/2012

BuyableStandardWheels = {};

function BuyableStandardWheels.prerequisitesPresent(specializations)
    return true;
end;

function BuyableStandardWheels:load(xmlFile)

	self.standardWheelTriggerCallback = BuyableStandardWheels.standardWheelTriggerCallback;
	self.wheelDelete = BuyableStandardWheels.wheelDelete;
	self.assembleWheels = SpecializationUtil.callSpecializationsFunction("assembleWheels");
	self.disassembleWheels = SpecializationUtil.callSpecializationsFunction("disassembleWheels");

	self.checkString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.standardWheels#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.standardWheels#activationTrigger"));


	addTrigger(self.usingTrigger, "standardWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);

	self.belts = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.standardWheels#belts"));

    self.buyableStandardWheels = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.standardWheels.wheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;

		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.transFix = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#transFix"));
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.buyableStandardWheels, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.standardWheelsUser = nil;

	self.savedVehicleID = -1;
	self.shouldUpdateLater = false;
end;

function BuyableStandardWheels:delete()
	if self.standardWheelsUser ~= nil then
		self.standardWheelsUser:onDisassembling(self);
		for _,standardWheel in pairs(self.buyableStandardWheels) do
			delete(standardWheel.node);
			delete(standardWheel.collision);
		end;
		self.buyableStandardWheels = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableStandardWheels:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableStandardWheels:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local standardWheelsUser = networkGetObject(id);
		if standardWheelsUser ~= nil then
			self:assembleWheels(standardWheelsUser, true);
		else
			self.savedVehicleID = id;
			self.shouldUpdateLater = true;
		end;
	end;
end;

function BuyableStandardWheels:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.standardWheelsUser ~= nil then
		idToWrite = networkGetObjectId(self.standardWheelsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableStandardWheels:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableStandardWheels:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableStandardWheels:update(dt)
end;

function BuyableStandardWheels:updateTick(dt)

	-- try to load saved standardWheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then
				self:assembleWheels(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;

	-- Attach wheels on client if not attached on load
	if self.shouldUpdateLater then
		local standardWheelsUser = networkGetObject(self.savedVehicleID);
		if standardWheelsUser ~= nil then
			self:assembleWheels(standardWheelsUser, true);
			self.shouldUpdateLater = false;
			self.savedVehicleID = -1;
		end;
	end;
end;

function BuyableStandardWheels:draw()
end;

function BuyableStandardWheels:standardWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter or onStay then
			if vehicle.rootNode == otherId then
				if vehicle.buyableWheels ~= nil then
					if vehicle.buyableWheels.checkString == self.checkString then
						if self.standardWheelsUser ~= nil then
							if self.standardWheelsUser == vehicle then
								vehicle.buyableWheels.wheelsInRange = self;
							end;
						else
							vehicle.buyableWheels.wheelsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then
			if otherId == vehicle.rootNode then
				if vehicle.buyableWheels ~= nil then
					vehicle.buyableWheels.wheelsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableStandardWheels:assembleWheels(vehicle, noEventSend)
	BuyableWheelsAttachEvent.sendEvent(self, vehicle, noEventSend);

	if self.belts ~= nil then
		setVisibility(self.belts, false);
	end;

	if self.standardWheelsUser == nil then
		if vehicle.buyableWheels ~= nil then
			self.standardWheelsUser = vehicle;
			for l,standardWheel in pairs(self.buyableStandardWheels) do
				if l == 1 then
					unlink(standardWheel.node);
					unlink(standardWheel.transFix);
					link(vehicle.wheels[1].driveNode, standardWheel.node);
				elseif l == 2 then
					unlink(standardWheel.node);
					unlink(standardWheel.transFix);
					link(vehicle.wheels[2].driveNode, standardWheel.node);
				elseif l == 3 then
					unlink(standardWheel.node);
					unlink(standardWheel.transFix);
					link(vehicle.wheels[3].driveNode, standardWheel.node);
				elseif l == 4 then
					unlink(standardWheel.node);
					unlink(standardWheel.transFix);
					link(vehicle.wheels[4].driveNode, standardWheel.node);
				end;
			end;
			self.standardWheelsUser:onAssembling(self);
		end;
	end;
end;

function BuyableStandardWheels:disassembleWheels(vehicle, noEventSend)
	BuyableWheelsDetachEvent.sendEvent(self, vehicle, noEventSend);

	self.standardWheelsUser:onDisassembling(self);

	if self.belts ~= nil then
		setVisibility(self.belts, true);
	end;

	for k,wheel in pairs(self.buyableStandardWheels) do
		unlink(wheel.transFix);
		unlink(wheel.node);
		setRotation(wheel.transFix, 0,0,0);
		setTranslation(wheel.transFix, 0,0,0);
		link(wheel.savePosition, wheel.transFix);
		link(wheel.transFix, wheel.node);
	end;
	self.standardWheelsUser = nil;
end;


function BuyableStandardWheels:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoords");
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};
		end;
	end;

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableStandardWheels:getSaveAttributesAndNodes(nodeIdent)
    local attributes = nil;

	if self.standardWheelsUser ~= nil then
		local x,y,z = getWorldTranslation(self.standardWheelsUser.rootNode);
		attributes = 'attacherCoords="'.. x .. " " .. y .. " " .. z ..'"';
	end;

    return attributes, nil;
end;
