--
-- RotorSpreader
--
--
-- @author  Manuel Leithner
-- @date  28/06/13
--
-- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.

RotorSpreader = {};
RotorSpreader.STATE_DISABLED = 0;
RotorSpreader.STATE_STARTING = 1;
RotorSpreader.STATE_RUNNING  = 2;
RotorSpreader.STATE_STOP     = 3;
RotorSpreader.STATE_STOPPING = 4;

function RotorSpreader.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Sprayer, specializations) and SpecializationUtil.hasSpecialization(Cylindered, specializations);
end;

function RotorSpreader:load(xmlFile)

    self.animation = {};
    self.animation.state = RotorSpreader.STATE_DISABLED;
    self.animation.startWeight = 1;
    self.animation.runWeight = 0;
    self.animation.endWeight = 0;
    self.animation.crossFadeTime = 200;
    self.animation.runThreshold = 0.9;

    self.animation.rootNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.rotorAnimation#rootNode"));
    self.animation.startClipSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.rotorAnimation#startClipSpeed", 1));
    self.animation.runClipSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.rotorAnimation#runClipSpeed", 1));
    self.animation.endClipSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.rotorAnimation#endClipSpeed", 1));
    self.animation.animCharSet = getAnimCharacterSet(self.animation.rootNode);
    if self.animation.animCharSet ~= 0 then
        local startClip = getAnimClipIndex(self.animation.animCharSet, getXMLString(xmlFile, "vehicle.rotorAnimation#startClip"));
        local runClip = getAnimClipIndex(self.animation.animCharSet, getXMLString(xmlFile, "vehicle.rotorAnimation#runClip"));
        local endClip = getAnimClipIndex(self.animation.animCharSet, getXMLString(xmlFile, "vehicle.rotorAnimation#endClip"));
        assignAnimTrackClip(self.animation.animCharSet, 0, startClip);
        assignAnimTrackClip(self.animation.animCharSet, 1, runClip);
        assignAnimTrackClip(self.animation.animCharSet, 2, endClip);
        self.animation.startDuration = getAnimClipDuration(self.animation.animCharSet, startClip);
        self.animation.runDuration = getAnimClipDuration(self.animation.animCharSet, runClip);
        self.animation.endDuration = getAnimClipDuration(self.animation.animCharSet, endClip);
        setAnimTrackSpeedScale(self.animation.animCharSet, 0, self.animation.startClipSpeed);
        setAnimTrackSpeedScale(self.animation.animCharSet, 1, self.animation.runClipSpeed);
        setAnimTrackSpeedScale(self.animation.animCharSet, 2, self.animation.endClipSpeed);
        setAnimTrackLoopState(self.animation.animCharSet, 0, false);
        setAnimTrackLoopState(self.animation.animCharSet, 1, true);
        setAnimTrackLoopState(self.animation.animCharSet, 2, false);
        setAnimTrackTime(self.animation.animCharSet, 0, 0);
        setAnimTrackTime(self.animation.animCharSet, 1, 0);
        setAnimTrackTime(self.animation.animCharSet, 2, 0);
        setAnimTrackBlendWeight(self.animation.animCharSet, 0, self.animation.startWeight);
        setAnimTrackBlendWeight(self.animation.animCharSet, 1, self.animation.runWeight);
        setAnimTrackBlendWeight(self.animation.animCharSet, 2, self.animation.endWeight);
    end;
    self.initializeAnimation = true;
    self.isAnimationRunning = false;
end;

function RotorSpreader:delete()
end;

function RotorSpreader:mouseEvent(posX, posY, isDown, isUp, button)
end;

function RotorSpreader:keyEvent(unicode, sym, modifier, isDown)
end;

function RotorSpreader:update(dt)
    if self.initializeAnimation then
        enableAnimTrack(self.animation.animCharSet, 0);
        setAnimTrackTime(self.animation.animCharSet, 0, 0, true);
        disableAnimTrack(self.animation.animCharSet, 0);
        self.initializeAnimation = false;
    end;

    if self.animation.state ~= STATE_DISABLED then
        if self.animation.state == RotorSpreader.STATE_STARTING then
            self.animation.startWeight = math.min(self.animation.startWeight + dt/self.animation.crossFadeTime, 1);
            self.animation.endWeight = 1-self.animation.startWeight;

            if getAnimTrackTime(self.animation.animCharSet, 0) >= self.animation.startDuration then
                self.animation.runWeight = 1;
                self.animation.startWeight = 0;
                self.animation.endWeight = 0;
                setAnimTrackTime(self.animation.animCharSet, 0, 0);
                setAnimTrackTime(self.animation.animCharSet, 1, 0);
                enableAnimTrack(self.animation.animCharSet, 1);
                disableAnimTrack(self.animation.animCharSet, 0);
                self.animation.state = RotorSpreader.STATE_RUNNING;
            end;
        elseif self.animation.state == RotorSpreader.STATE_STOP then
            local runTime = getAnimTrackTime(self.animation.animCharSet, 1);
            if runTime >= self.animation.runDuration*self.animation.runThreshold then
                setAnimTrackTime(self.animation.animCharSet, 2, 0);
                enableAnimTrack(self.animation.animCharSet, 2);
                self.animation.state = RotorSpreader.STATE_STOPPING
            end;
        elseif self.animation.state == RotorSpreader.STATE_STOPPING then
            self.animation.endWeight = math.min(self.animation.endWeight + dt/self.animation.crossFadeTime, 1);
            self.animation.runWeight = 1-self.animation.endWeight;

            if getAnimTrackTime(self.animation.animCharSet, 2) >= self.animation.endDuration then
                disableAnimTrack(self.animation.animCharSet, 2);
                disableAnimTrack(self.animation.animCharSet, 1);
                self.animation.state = RotorSpreader.STATE_DISABLED;
                self.animation.runWeight = 0;
            end;
        end;
        setAnimTrackBlendWeight(self.animation.animCharSet, 0, self.animation.startWeight);
        setAnimTrackBlendWeight(self.animation.animCharSet, 1, self.animation.runWeight);
        setAnimTrackBlendWeight(self.animation.animCharSet, 2, self.animation.endWeight);
    end;
end;

function RotorSpreader:updateTick(dt)
end;

function RotorSpreader:draw()
    if self.animation.state == RotorSpreader.STATE_STARTING then
        g_currentMission:addExtraPrintText(g_i18n:getText("RotorSpreader_startingRotor"));
    end;
    if self.animation.state == RotorSpreader.STATE_STOP or self.animation.state == RotorSpreader.STATE_STOPPING then
        g_currentMission:addExtraPrintText(g_i18n:getText("RotorSpreader_stoppingRotor"));
    end;
end;

function RotorSpreader:onDeactivate()
    if self.animation.state == RotorSpreader.STATE_RUNNING then
        self.animation.state = RotorSpreader.STATE_STOP;
    end;
    if self.animation.state == RotorSpreader.STATE_STARTING then
        self.animation.state = RotorSpreader.STATE_STOPPING;
    end;
end;

function RotorSpreader:onTurnedOn()
    setAnimTrackTime(self.animation.animCharSet, 0, 0);
    enableAnimTrack(self.animation.animCharSet, 0);
    self.animation.state = RotorSpreader.STATE_STARTING;
end;

function RotorSpreader:onTurnedOff()
    self.animation.state = RotorSpreader.STATE_STOP;
end;